document.addEventListener('DOMContentLoaded', function() {
    const uploadBtn = document.getElementById('uploadBtn');
    const resumeUpload = document.getElementById('resumeUpload');
    const uploadStatus = document.getElementById('uploadStatus');
    const infoSection = document.getElementById('infoSection');
    const saveBtn = document.getElementById('saveBtn');
    const fillBtn = document.getElementById('fillBtn');
    const addExperience = document.getElementById('addExperience');
    const experienceList = document.getElementById('experienceList');

    loadSavedData();

    uploadBtn.addEventListener('click', () => {
        resumeUpload.click();
    });

    resumeUpload.addEventListener('change', handleFileUpload);
    saveBtn.addEventListener('click', saveData);
    fillBtn.addEventListener('click', fillForm);
    addExperience.addEventListener('click', addExperienceItem);

    function handleFileUpload(event) {
        const file = event.target.files[0];
        if (!file) return;

        showStatus('正在解析简历...', 'info');
        
        const reader = new FileReader();
        reader.onload = function(e) {
            try {
                parseResumeContent(e.target.result, file.name);
            } catch (error) {
                showStatus('文件解析失败，请检查文件格式', 'error');
            }
        };
        
        if (file.type.includes('pdf')) {
            reader.readAsArrayBuffer(file);
        } else {
            reader.readAsText(file, 'utf-8');
        }
    }

    function parseResumeContent(content, filename) {
        setTimeout(() => {
            const mockData = {
                name: '张三',
                phone: '13800138000',
                email: 'zhangsan@email.com',
                school: '北京大学',
                major: '计算机科学与技术',
                education: '本科',
                experiences: [
                    {
                        company: '腾讯科技',
                        position: '前端开发实习生',
                        duration: '2023.06-2023.09',
                        description: '负责微信小程序开发，参与用户界面优化项目'
                    }
                ],
                awards: '国家奖学金、ACM程序设计竞赛二等奖、优秀学生干部'
            };
            
            fillFormData(mockData);
            showStatus('简历解析成功！', 'success');
            infoSection.style.display = 'block';
        }, 1500);
    }

    function fillFormData(data) {
        document.getElementById('name').value = data.name || '';
        document.getElementById('phone').value = data.phone || '';
        document.getElementById('email').value = data.email || '';
        document.getElementById('school').value = data.school || '';
        document.getElementById('major').value = data.major || '';
        document.getElementById('education').value = data.education || '';
        document.getElementById('awards').value = data.awards || '';
        
        experienceList.innerHTML = '';
        if (data.experiences) {
            data.experiences.forEach(exp => addExperienceItem(exp));
        }
    }

    function addExperienceItem(data = {}) {
        const expDiv = document.createElement('div');
        expDiv.className = 'experience-item';
        expDiv.innerHTML = `
            <button class="remove-btn" type="button">删除</button>
            <input type="text" placeholder="公司名称" value="${data.company || ''}" data-field="company">
            <input type="text" placeholder="职位" value="${data.position || ''}" data-field="position">
            <input type="text" placeholder="时间 (如: 2023.06-2023.09)" value="${data.duration || ''}" data-field="duration">
            <textarea placeholder="工作描述" data-field="description" rows="3">${data.description || ''}</textarea>
        `;
        
        expDiv.querySelector('.remove-btn').addEventListener('click', () => {
            expDiv.remove();
        });
        
        experienceList.appendChild(expDiv);
    }

    function saveData() {
        const data = {
            name: document.getElementById('name').value,
            phone: document.getElementById('phone').value,
            email: document.getElementById('email').value,
            school: document.getElementById('school').value,
            major: document.getElementById('major').value,
            education: document.getElementById('education').value,
            awards: document.getElementById('awards').value,
            experiences: []
        };
        
        experienceList.querySelectorAll('.experience-item').forEach(item => {
            const exp = {};
            item.querySelectorAll('[data-field]').forEach(field => {
                exp[field.dataset.field] = field.value;
            });
            data.experiences.push(exp);
        });
        
        chrome.storage.local.set({ resumeData: data }, () => {
            showStatus('信息保存成功！', 'success');
        });
    }

    function loadSavedData() {
        chrome.storage.local.get(['resumeData'], (result) => {
            if (result.resumeData) {
                fillFormData(result.resumeData);
                infoSection.style.display = 'block';
            }
        });
    }

    function fillForm() {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            chrome.tabs.sendMessage(tabs[0].id, { 
                action: 'fillForm',
                data: getCurrentFormData()
            }, (response) => {
                if (response && response.success) {
                    showStatus('表单填写成功！', 'success');
                } else {
                    showStatus('未找到可填写的表单字段', 'error');
                }
            });
        });
    }

    function getCurrentFormData() {
        const data = {
            name: document.getElementById('name').value,
            phone: document.getElementById('phone').value,
            email: document.getElementById('email').value,
            school: document.getElementById('school').value,
            major: document.getElementById('major').value,
            education: document.getElementById('education').value,
            awards: document.getElementById('awards').value,
            experiences: []
        };
        
        experienceList.querySelectorAll('.experience-item').forEach(item => {
            const exp = {};
            item.querySelectorAll('[data-field]').forEach(field => {
                exp[field.dataset.field] = field.value;
            });
            data.experiences.push(exp);
        });
        
        return data;
    }

    function showStatus(message, type) {
        uploadStatus.textContent = message;
        uploadStatus.className = `status-message ${type}`;
        
        if (type === 'success') {
            setTimeout(() => {
                uploadStatus.style.display = 'none';
            }, 3000);
        }
    }
});