chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'fillForm') {
        const result = fillFormFields(request.data);
        sendResponse({ success: result });
    }
});

function fillFormFields(data) {
    let filledCount = 0;
    
    const fieldMappings = {
        name: ['姓名', '真实姓名', '名字', '用户名', 'name', 'realname', 'username', 'full_name', 'fullname'],
        phone: ['电话', '手机', '联系方式', '手机号', '联系电话', 'phone', 'mobile', 'telephone', 'contact', 'cellphone'],
        email: ['邮箱', '电子邮箱', '电邮', 'email', 'e-mail', 'mail', 'email_address'],
        school: ['学校', '毕业院校', '院校', '大学', 'school', 'university', 'college', 'institution'],
        major: ['专业', '所学专业', '专业名称', 'major', 'specialty', 'discipline'],
        education: ['学历', '教育程度', '最高学历', 'education', 'degree', 'educational_level'],
        awards: ['获奖情况', '荣誉奖项', '获奖经历', 'awards', 'honors', 'achievements']
    };
    
    for (const [key, keywords] of Object.entries(fieldMappings)) {
        if (data[key]) {
            const fields = findFieldsByKeywords(keywords);
            fields.forEach(field => {
                if (fillField(field, data[key])) {
                    filledCount++;
                }
            });
        }
    }
    
    if (data.experiences && data.experiences.length > 0) {
        const exp = data.experiences[0];
        
        const experienceKeywords = {
            company: ['实习公司', '公司名称', '工作单位', '实习单位', 'company', 'organization', 'employer'],
            position: ['实习职位', '岗位', '职务', '实习岗位', 'position', 'job_title', 'title'],
            duration: ['实习时间', '工作时间', '时间', 'duration', 'time_period', 'period'],
            description: ['工作描述', '实习内容', '工作内容', '岗位描述', 'description', 'job_description', 'content']
        };
        
        for (const [key, keywords] of Object.entries(experienceKeywords)) {
            if (exp[key]) {
                const fields = findFieldsByKeywords(keywords);
                fields.forEach(field => {
                    if (fillField(field, exp[key])) {
                        filledCount++;
                    }
                });
            }
        }
    }
    
    return filledCount > 0;
}

function findFieldsByKeywords(keywords) {
    const fields = [];
    
    const inputs = document.querySelectorAll('input[type="text"], input[type="email"], input[type="tel"], textarea, select');
    
    inputs.forEach(input => {
        const fieldText = getFieldContext(input).toLowerCase();
        
        for (const keyword of keywords) {
            if (fieldText.includes(keyword.toLowerCase())) {
                fields.push(input);
                break;
            }
        }
    });
    
    return fields;
}

function getFieldContext(element) {
    let context = '';
    
    if (element.placeholder) {
        context += ' ' + element.placeholder;
    }
    
    if (element.name) {
        context += ' ' + element.name;
    }
    
    if (element.id) {
        context += ' ' + element.id;
    }
    
    const label = findAssociatedLabel(element);
    if (label) {
        context += ' ' + label.textContent;
    }
    
    const parentText = getParentText(element);
    if (parentText) {
        context += ' ' + parentText;
    }
    
    return context;
}

function findAssociatedLabel(element) {
    if (element.id) {
        const label = document.querySelector(`label[for="${element.id}"]`);
        if (label) return label;
    }
    
    let parent = element.parentElement;
    while (parent && parent.tagName !== 'FORM') {
        const label = parent.querySelector('label');
        if (label) return label;
        parent = parent.parentElement;
    }
    
    return null;
}

function getParentText(element) {
    let parent = element.parentElement;
    let depth = 0;
    
    while (parent && depth < 3) {
        const text = Array.from(parent.childNodes)
            .filter(node => node.nodeType === Node.TEXT_NODE)
            .map(node => node.textContent.trim())
            .join(' ');
            
        if (text.length > 0) {
            return text;
        }
        
        parent = parent.parentElement;
        depth++;
    }
    
    return '';
}

function fillField(field, value) {
    if (field.type === 'select-one' || field.tagName === 'SELECT') {
        return fillSelectField(field, value);
    } else {
        return fillTextualField(field, value);
    }
}

function fillTextualField(field, value) {
    if (field.value !== '' && !confirm(`字段"${getFieldContext(field).slice(0, 20)}..."已有内容，是否覆盖？`)) {
        return false;
    }
    
    field.value = value;
    field.dispatchEvent(new Event('input', { bubbles: true }));
    field.dispatchEvent(new Event('change', { bubbles: true }));
    
    addVisualFeedback(field);
    return true;
}

function fillSelectField(select, value) {
    const options = Array.from(select.options);
    
    const exactMatch = options.find(option => 
        option.value.toLowerCase() === value.toLowerCase() || 
        option.text.toLowerCase() === value.toLowerCase()
    );
    
    if (exactMatch) {
        select.value = exactMatch.value;
        select.dispatchEvent(new Event('change', { bubbles: true }));
        addVisualFeedback(select);
        return true;
    }
    
    const partialMatch = options.find(option => 
        option.value.toLowerCase().includes(value.toLowerCase()) || 
        option.text.toLowerCase().includes(value.toLowerCase()) ||
        value.toLowerCase().includes(option.text.toLowerCase())
    );
    
    if (partialMatch) {
        select.value = partialMatch.value;
        select.dispatchEvent(new Event('change', { bubbles: true }));
        addVisualFeedback(select);
        return true;
    }
    
    return false;
}

function addVisualFeedback(field) {
    const originalBorder = field.style.border;
    field.style.border = '2px solid #4CAF50';
    field.style.transition = 'border 0.3s ease';
    
    setTimeout(() => {
        field.style.border = originalBorder;
    }, 2000);
}

window.addEventListener('load', () => {
    const style = document.createElement('style');
    style.textContent = `
        .auto-fill-indicator {
            position: relative;
        }
        .auto-fill-indicator::after {
            content: '✓';
            position: absolute;
            right: 5px;
            top: 50%;
            transform: translateY(-50%);
            color: #4CAF50;
            font-weight: bold;
            pointer-events: none;
        }
    `;
    document.head.appendChild(style);
});